//using UnityEngine;
using System;
using System.Runtime.InteropServices;
using System.Text;
using System.Collections;
using System.Collections.Generic;

namespace Sony
{
	namespace NP
	{
		public class Matching
		{
			static bool hosting = false;

			// Session type, i.e. public or private.
			// IMPORTANT, These must exactly match the SceToolkitNpSessionCreateFlag defines in SCE npToolkit matching_defines.h
			public enum EnumSessionType
			{
				SESSION_TYPE_DEFAULT = 0,		// If not setting public/private then we need to supply slot info. NOTE, this is currently un-supported.
				SESSION_TYPE_PUBLIC =  (1<<2),	// This flag specifies that all the slots in the session are available to the public.
				SESSION_TYPE_PRIVATE = (1<<3),	// This flag specifies that all the slots in the session are reserved for private players or friends. If this flag is set, then the session will not be visible to other users during searches.
			};

			// Session creation flags. NOTE that for network play you need, at a minimum, to specify CREATE_SIGNALING_MESH_SESSION.
			// IMPORTANT, These must exactly match the SceToolkitNpSessionCreateFlag defines in SCE npToolkit matching_defines.h
			[Flags] public enum FlagSessionCreate
			{
				CREATE_DEFAULT = 0,
				CREATE_SIGNALING_MESH_SESSION = 	(1<<2),	// This flag specifies that the session supports signaling.
                //CREATE_SIGNALING_STAR_SESSION = 	(1<<3),	// (Currently not supported) This flag specifies that the session supports signaling.
				CREATE_PASSWORD_SESSION = 			(1<<4),	// This flag specifies that the session is password protected. 
				CREATE_ALLOW_BLOCK_LIST_SESSION = 	(1<<5),	// This flag specifies that the current session can be joined by a blocked user. This is not allowed by default.  
				CREATE_HOST_MIGRATION_SESSION =		(1<<6),	// This flag specifies that the session should be allowed to migrate when the owner quits. The ability to grant room ownership is not supported by default.
				CREATE_NAT_RESTRICTED_SESSION =		(1<<7),	// This flag specifies that users who cannot establish P2P connections are not allowed to join the session. 
			};

			// Valid attribute types.
			// IMPORTANT, These must exactly match the SceToolkitNpSessionAttributeType defines in SCE npToolkit matching_defines.h
			public enum EnumAttributeType
			{
				SESSION_SEARCH_ATTRIBUTE =			(1<<1), // This specifies that the session attribute is used as a search filter. 
				SESSION_EXTERNAL_ATTRIBUTE =		(1<<2), // This specifies that the session attribute is used as external session room data.
				SESSION_INTERNAL_ATTRIBUTE =        (1<<3), // This specifies that the session attribute is used as internal session room data. 
                SESSION_MEMBER_ATTRIBUTE =          (1<<4), // This specifies that the session attribute is used as session member data.
            };

			// Valid attribute value types.
			// IMPORTANT, These must exactly match the SceToolkitNpSessionAttributeValueType defines in SCE npToolkit matching_defines.h
			public enum EnumAttributeValueType
            {
                SESSION_ATTRIBUTE_VALUE_INT =      (1<<1), // The value of this attribute is an integer.
                SESSION_ATTRIBUTE_VALUE_BINARY =   (1<<2), // The value of this attribute is a binary value.
            };

			// Maximum data sizes for attributes that contain binary data.
			// IMPORTANT, These must exactly match the SceToolkitNpSessionAttributeMaxSize defines in SCE npToolkit matching_defines.h
			public enum EnumAttributeMaxSize
            {
                SESSION_ATTRIBUTE_MAX_SIZE_12 =    (1<<1), // The value of this attribute is a binary value with a maximum size of 12 characters. 
                SESSION_ATTRIBUTE_MAX_SIZE_28 =    (1<<2), // The value of this attribute is a binary value with a maximum size of 28 characters.
                SESSION_ATTRIBUTE_MAX_SIZE_60 =    (1<<3), // The value of this attribute is a binary value with a maximum size of 60 characters. 
                SESSION_ATTRIBUTE_MAX_SIZE_124 =   (1<<4), // The value of this attribute is a binary value with a maximum size of 124 characters. 
                SESSION_ATTRIBUTE_MAX_SIZE_252 =   (1<<5), // The value of this attribute is a binary value with a maximum size of 252 characters. 
            }

			// Valid attributes search opearator values.
			// IMPORTANT, These must exactly match the SceNpMatching2Operator defines in SCE np_matching2.h
			public enum EnumSearchOperators
            {
                MATCHING_OPERATOR_INVALID,
                MATCHING_OPERATOR_EQ,
                MATCHING_OPERATOR_NE,
                MATCHING_OPERATOR_LT,
                MATCHING_OPERATOR_LE,
                MATCHING_OPERATOR_GT,
                MATCHING_OPERATOR_GE,
            }
			
			// Flags defining the type of member, i.e. owner and/pr self.
			// IMPORTANT, These must exactly match the SceToolkitNpSessionMemberFlag defines in SCE npToolkit matching_defines.h
			[Flags] public enum FlagMemberType
			{
				MEMBER_OWNER = (1<<1),	// This flag specifies that the current member is the session owner.
				MEMBER_MYSELF = (1<<2),	// This flag specifies that the member is the user themselves (local).
			};

			// Structure for passing session attributes to NP.
			[StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi, Pack = 0)]
			public class SessionAttribute
			{
                IntPtr _name;
                IntPtr _binValue;
                public int intValue;
                public EnumSearchOperators searchOperator;

				public string name
				{
					//get { return Marshal.PtrToStringAnsi(_name); }
					set { _name = Marshal.StringToCoTaskMemAnsi(value); }
				}

				public string binValue
				{
					//get { return Marshal.PtrToStringAnsi(_binValue); }
					set { _binValue = Marshal.StringToCoTaskMemAnsi(value); }
				}

				~SessionAttribute()
				{
					Marshal.FreeCoTaskMem(_binValue);
				}
			};

			// Structure for passing session attribute modifications to NP.
			[StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi, Pack = 0)]
			public class ModifySessionAttribute
			{
				IntPtr _name;
				IntPtr _binValue;
				public int intValue;

				public string name
				{
					//get { return Marshal.PtrToStringAnsi(_name); }
					set { _name = Marshal.StringToCoTaskMemAnsi(value); }
				}
				
				public string binValue
				{
					//get { return Marshal.PtrToStringAnsi(_binValue); }
					set { _binValue = Marshal.StringToCoTaskMemAnsi(value); }
				}
				
				~ModifySessionAttribute()
				{
					Marshal.FreeCoTaskMem(_binValue);
				}
			};

			// Structure containing all information pertaining to a session, typically the result of a session search.
			public struct Session
			{
				public SessionInfo sessionInfo;							// Session information.
				public SessionAttributeInfo[] sessionAttributes;		// Session attributes.
				public SessionMemberInfo[] members;						// Members in this session.
				public List<SessionAttributeInfo[]> memberAttributes;	// Attributes for each member.
			}
			
			// Structure for retrieving session information.
			[StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi, Pack = 0)]
			public struct SessionInfo
			{
				IntPtr _sessionName;
				public int sessionID;				// The ID of this session, use this ID when joining a session.
				public int maxMembers;				// The maximum numbers of members in the session.
				public int numMembers;				// The current number of members in the session.
				public int numSessionAttributes;	// The number of attributes in the session. 
				public int reservedSlots;			// The number of slots that are reserved for a friend in a session.
				public int openSlots;				// The number of slots that are open to the public.
				public int worldId;					// The world ID for a session. 
				public int serverId;				// The server ID for a session.
				public int matchingContext;			// The Matching2 context ID. This should be used when using the %NP Matching2 library chat function to send messages to members in the current session.
				public UInt64 roomId;				// The ID of the current session.

				public string sessionName			// Session name.
				{
					get { return Marshal.PtrToStringAnsi(_sessionName); }
				}
			};

			// Structure for retrieving session attribute information.
			[StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi, Pack = 0)]
			public struct SessionAttributeInfo
			{
				IntPtr _attribute;					
				public EnumAttributeType attributeType;		// The type of the attribute.
				public EnumSearchOperators searchOperator;	// The search operator if the session attribute type is SCE_SESSION_SEARCH_ATTRIBUTE_*. This is used when filtering the sessions.
				public EnumAttributeMaxSize maxSize;		// The size of the data in attributeValue.
				public EnumAttributeValueType attributeValueType;	// The type of value contained in attributeValue, SCE_SESSION_ATTRIBUTE_VALUE_* flags.
				public int attributeIntValue;				// The integer value of a session attribute. 
				IntPtr _attributeBinValue;					// The binary data of a session attribute.

				public string attributeName					// The name of the session attribute.
				{
					get { return Marshal.PtrToStringAnsi(_attribute); }
				}
				
				public string attributeBinValue				// The name of the session attribute.
				{
					get { return Marshal.PtrToStringAnsi(_attributeBinValue); }
				}
			};

			// Structure for retrieving member info from an active session.
			[StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi, Pack = 0)]
			public struct SessionMemberInfo
			{
				IntPtr _npID;					// NpID, contains data corresponding to the SceNpID structure, can be used as id to send to when messaging.
				int npIDSize;					// Size of the npID byte array.
				IntPtr _npOnlineID;
				public int memberId;			// The room member ID of the member.
				public int natType;				// The member's NAT Type.
				public FlagMemberType memberFlag;	// Specifies whether this member is the owner, another member or the current user.
				public UInt64 joinDate;			// The date and time the user joined the session.
				public int addr;				// The IP address. 
				public int port;				// The port number.

				public byte[] npID
				{
					get
					{
						byte[] bytes = new byte[npIDSize];
						Marshal.Copy(_npID, bytes, 0, npIDSize);
						return bytes;
					}
				}
				public string npOnlineID		// Online ID.
				{
					get { return Marshal.PtrToStringAnsi(_npOnlineID); }
				}
			};

			// Session status...
            [DllImport("UnityNpToolkit")]
            private static extern bool PrxSessionIsBusy();

			[DllImport("UnityNpToolkit")]
			private static extern bool PrxSessionGetLastError(out ResultCode result);
			
			public static bool GetLastError(out ResultCode result)
			{
				PrxSessionGetLastError(out result);
				return result.lastError == ErrorCode.NP_OK;
			}

			[DllImport("UnityNpToolkit")]
			private static extern bool PrxSessionInSession();

			// Game session attributes...
			[DllImport("UnityNpToolkit")]
            private static extern ErrorCode PrxSessionClearAttributeDefinitions();
            [DllImport("UnityNpToolkit", CharSet = CharSet.Ansi)]
			private static extern ErrorCode PrxSessionAddAttributeDefinition(string name, EnumAttributeType type, EnumAttributeValueType valueType, EnumAttributeMaxSize maxSize);
            [DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxSessionRegisterAttributeDefinitions();

			// Session creation...
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxSessionClearAttributes();
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxSessionAddAttribute(SessionAttribute sessionAttribute);
			[DllImport("UnityNpToolkit", CharSet = CharSet.Ansi)]
			private static extern ErrorCode PrxSessionCreateSession(string name, int serverID, int worldID, int numSlots, string password, FlagSessionCreate creationFlags, EnumSessionType sessionType, string ps4SessionStatus);
			[DllImport("UnityNpToolkit", CharSet = CharSet.Ansi)]
			private static extern ErrorCode PrxSessionCreateFriendsSession(string name, int serverID, int worldID, int numSlots, int numFriendSlots, string password, FlagSessionCreate creationFlags, string ps4SessionStatus);

			// Session joining...
			[DllImport("UnityNpToolkit", CharSet = CharSet.Ansi)]
			private static extern ErrorCode PrxSessionJoinSession(int sessionID, string password);
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxSessionJoinInvitedSession();

			// Session Modification...
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxSessionModifyClearAttributes();
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxSessionModifyAddAttribute(ModifySessionAttribute attribute);
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxSessionModifySession(EnumAttributeType attributeType);

			// Active session info...
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxGetSessionInfo(out SessionInfo info);

			// Active session attributes...
			[DllImport("UnityNpToolkit")]
			private static extern void PrxLockSessionAttributeList();
			[DllImport("UnityNpToolkit")]
			private static extern void PrxUnlockSessionAttributeList();
			[DllImport("UnityNpToolkit")]
			private static extern int PrxGetSessionAttributeListCount();
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxGetSessionAttributeInfo(int index, out SessionAttributeInfo info);

			// Active session member info...
			[DllImport("UnityNpToolkit")]
			private static extern void PrxLockSessionMemberList();
			[DllImport("UnityNpToolkit")]
			private static extern void PrxUnlockSessionMemberList();
			[DllImport("UnityNpToolkit")]
			private static extern int PrxGetSessionMemberListCount();
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxGetSessionMemberInfo(int index, out SessionMemberInfo info);

			// Active session member attributes...
			[DllImport("UnityNpToolkit")]
			private static extern void PrxLockSessionMemberAttributeList();
			[DllImport("UnityNpToolkit")]
			private static extern void PrxUnlockSessionMemberAttributeList();
			[DllImport("UnityNpToolkit")]
			private static extern int PrxGetSessionMemberAttributeListCount(int memberIndex);
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxGetSessionMemberAttributeInfo(int memberIndex, int index, out SessionAttributeInfo info);

			// Leaving a session...
            [DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxSessionLeaveSession();

			// Session searching...
            [DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxSessionFind(int serverID, int worldID);
            [DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxSessionFindFriends(int serverID, int worldID);
            [DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxSessionFindRegional(int serverID, int worldID);

			// Found session info...
			[DllImport("UnityNpToolkit")]
			private static extern void PrxLockFoundSessionList();
			[DllImport("UnityNpToolkit")]
			private static extern void PrxUnlockFoundSessionList();
			[DllImport("UnityNpToolkit")]
			private static extern int PrxGetFoundSessionListCount();
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxGetFoundSessionInfo(int sessionIndex, out SessionInfo info);
			
			// Found session attributes
			[DllImport("UnityNpToolkit")]
			private static extern void PrxLockFoundSessionAttributeList();
			[DllImport("UnityNpToolkit")]
			private static extern void PrxUnlockFoundSessionAttributeList();
			[DllImport("UnityNpToolkit")]
			private static extern int PrxGetFoundSessionAttributeListCount(int sessionIndex);
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxGetFoundSessionAttributeInfo(int sessionIndex, int attributeIndex, out SessionAttributeInfo info);
			
			// Found session member info...
			[DllImport("UnityNpToolkit")]
			private static extern void PrxLockFoundSessionMemberList();
			[DllImport("UnityNpToolkit")]
			private static extern void PrxUnlockFoundSessionMemberList();
			[DllImport("UnityNpToolkit")]
			private static extern int PrxGetFoundSessionMemberListCount(int sessionIndex);
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxGetFoundSessionMemberInfo(int sessionIndex, int memberIndex, out SessionMemberInfo info);

			[DllImport("UnityNpToolkit", CharSet = CharSet.Ansi)]
			private static extern ErrorCode PrxInviteToSession(string text, int ps4NpIDCount);

			// Currently in a session?
            public static bool InSession
			{
				get { return PrxSessionInSession(); }
			}

			// Am I the host? i.e. was the active session created by me?
			public static bool IsHost
			{
				get { return hosting; }
			}

			// Is NP busy with a session request.
			public static bool SessionIsBusy
			{
				get { return PrxSessionIsBusy(); }
			}

			// Reset session attribute definitions.
			public static ErrorCode ClearAttributeDefinitions()
            {
                return PrxSessionClearAttributeDefinitions();
            }

			// Add an integer type session attribute definition.
			public static ErrorCode AddAttributeDefinitionInt(string name, EnumAttributeType type)
            {
                return PrxSessionAddAttributeDefinition(name, type, EnumAttributeValueType.SESSION_ATTRIBUTE_VALUE_INT, 0);
            }

			// Add a binary data type session attribute definition.
			public static ErrorCode AddAttributeDefinitionBin(string name, EnumAttributeType type, EnumAttributeMaxSize maxSize)
            {
                return PrxSessionAddAttributeDefinition(name, type, EnumAttributeValueType.SESSION_ATTRIBUTE_VALUE_BINARY, maxSize);
            }

			// Register the games session attribute definitions.
			public static ErrorCode RegisterAttributeDefinitions()
            {
                return PrxSessionRegisterAttributeDefinitions();
            }

			// Reset session creation attributes.
			public static ErrorCode ClearSessionAttributes()
			{
                return PrxSessionClearAttributes();
			}

			// Add a session creation attribute.
			public static ErrorCode AddSessionAttribute(SessionAttribute sessionAttribute)
			{
				ErrorCode ret = PrxSessionAddAttribute(sessionAttribute);
				return ret;
			}

			// Create a session.
			public static ErrorCode CreateSession(string name, int serverID, int worldID, int numSlots, string password, FlagSessionCreate creationFlags, EnumSessionType sessionType, string ps4SessionStatus)
			{
				return PrxSessionCreateSession(name, serverID, worldID, numSlots, password, creationFlags, sessionType, ps4SessionStatus);
			}

			// Create a session with slots reserved for friends.
			public static ErrorCode CreateFriendsSession(string name, int serverID, int worldID, int numSlots, int friendSlots, string password, FlagSessionCreate creationFlags, string ps4SessionStatus)
			{
				return PrxSessionCreateFriendsSession(name, serverID, worldID, numSlots, friendSlots, password, creationFlags, ps4SessionStatus);
			}

			// Join a session, takes a sessionID. sessionIDs are retrieved from the SessionInfo structures that are returned from a session search.
			public static ErrorCode JoinSession(int sessionID, string password)
			{
				return PrxSessionJoinSession(sessionID, password);
			}

			// Join an invited session.
			public static ErrorCode JoinInvitedSession()
			{
				return PrxSessionJoinInvitedSession();
			}
						
			// Reset session modification attributes.
			public static ErrorCode ClearModifySessionAttributes()
			{
				return PrxSessionModifyClearAttributes();
			}

			// Add an attribute for modifying a session.
			public static ErrorCode AddModifySessionAttribute(ModifySessionAttribute sessionAttribute)
			{
				ErrorCode ret = PrxSessionModifyAddAttribute(sessionAttribute);
				return ret;
			}

			// Modify a session, note that all attributes to be modified must share the same attribute type.
			public static ErrorCode ModifySession(EnumAttributeType attributeType)
			{
				return PrxSessionModifySession(attributeType);
			}

			// Get the information for the currently active session.
			public static Session GetSession()
			{
				Session session = new Session();
				
				// Session info.
				session.sessionInfo = new SessionInfo();
				PrxGetSessionInfo(out session.sessionInfo);

				PrxLockSessionAttributeList();

				// Session attributes.
				int attributeCount = PrxGetSessionAttributeListCount();
				session.sessionAttributes = new SessionAttributeInfo[attributeCount];
				for(int attributeIndex=0; attributeIndex<attributeCount; attributeIndex++)
				{
					PrxGetSessionAttributeInfo(attributeIndex,
					                           out session.sessionAttributes[attributeIndex]);
				}
				
				// Session members.
				int memberCount = PrxGetSessionMemberListCount();
				session.members = new SessionMemberInfo[memberCount];
				session.memberAttributes = new List<SessionAttributeInfo[]>();
				for(int memberIndex=0; memberIndex<memberCount; memberIndex++)
				{
					PrxGetSessionMemberInfo(memberIndex,
					                        out session.members[memberIndex]);
					
					// Member attributes.
					int memberAttributeCount = PrxGetSessionMemberAttributeListCount(memberIndex);
					SessionAttributeInfo[] attributes = new SessionAttributeInfo[memberAttributeCount];
					for(int memberAttrIndex=0; memberAttrIndex<memberAttributeCount; memberAttrIndex++)
					{
						PrxGetSessionMemberAttributeInfo(memberIndex,
					                                      memberAttrIndex,
					                                      out attributes[memberAttrIndex]);
					}
					session.memberAttributes.Add(attributes);
				}

				PrxUnlockSessionAttributeList();

				return session;
			}

			// Leave the active session.
			public static ErrorCode LeaveSession()
			{
				return PrxSessionLeaveSession();
			}

			// Find a session; use ClearSessionAttributes & AddSessionAttribute to build a list of attributes to use in the search before calling this method.
			public static ErrorCode FindSession(int serverID, int worldID)
            {
                return PrxSessionFind(serverID, worldID);
            }

			// Find a friend session; use ClearSessionAttributes & AddSessionAttribute to build a list of attributes to use in the search before calling this method.
			public static ErrorCode FindSessionFriends(int serverID, int worldID)
            {
                return PrxSessionFindFriends(serverID, worldID);
            }

			// Find a regional session; use ClearSessionAttributes & AddSessionAttribute to build a list of attributes to use in the search before calling this method.
			public static ErrorCode FindSessionRegional(int serverID, int worldID)
            {
                return PrxSessionFindRegional(serverID, worldID);
            }

			// Retrieve the list of found sessions.
			public static Session[] GetFoundSessionList()
			{
				PrxLockFoundSessionList();

				int count = PrxGetFoundSessionListCount();
				Session[] sessions = new Session[count];

				for(int sessionIndex=0; sessionIndex<count; sessionIndex++)
				{
					// Session info.
					sessions[sessionIndex].sessionInfo = new SessionInfo();
					PrxGetFoundSessionInfo(sessionIndex, out sessions[sessionIndex].sessionInfo);

					// Session attributes.
					int attributeCount = PrxGetFoundSessionAttributeListCount(sessionIndex);
					sessions[sessionIndex].sessionAttributes = new SessionAttributeInfo[attributeCount];
					for(int attributeIndex=0; attributeIndex<attributeCount; attributeIndex++)
					{
						PrxGetFoundSessionAttributeInfo(sessionIndex,
						                                attributeIndex,
						                                out sessions[sessionIndex].sessionAttributes[attributeIndex]);
					}
				}

				PrxUnlockFoundSessionList();

				return sessions;
			}

			// Invite users to a session.
			static public ErrorCode InviteToSession(string text, int npIDCount)
			{
				return PrxInviteToSession(text, npIDCount);
			}

			// Session event delegates.
			public static event Messages.EventHandler OnCreatedSession;
			public static event Messages.EventHandler OnFoundSessions;
			public static event Messages.EventHandler OnJoinedSession;
			public static event Messages.EventHandler OnJoinInvalidSession;
			public static event Messages.EventHandler OnUpdatedSession;
			public static event Messages.EventHandler OnLeftSession;
			public static event Messages.EventHandler OnSessionDestroyed;
			public static event Messages.EventHandler OnKickedOut;
			public static event Messages.EventHandler OnSessionError;

			// Message processing.
			public static bool ProcessMessage(Messages.PluginMessage msg)
			{
				switch (msg.type)
				{
					case Messages.MessageType.kNPToolKit_MatchingCreatedSession:
						hosting = true;
						if (OnCreatedSession != null) OnCreatedSession(msg);
						return true;
						
					case Messages.MessageType.kNPToolKit_MatchingFoundSessions:
						if (OnFoundSessions != null) OnFoundSessions(msg);
						return true;
						
					case Messages.MessageType.kNPToolKit_MatchingJoinInvalidSession:
						if (OnJoinInvalidSession != null) OnJoinInvalidSession(msg);
						return true;

					case Messages.MessageType.kNPToolKit_MatchingJoinedSession:
						hosting = false;
						if (OnJoinedSession != null) OnJoinedSession(msg);
						return true;
						
					case Messages.MessageType.kNPToolKit_MatchingUpdatedSession:
						if (OnUpdatedSession != null) OnUpdatedSession(msg);
						return true;
						
					case Messages.MessageType.kNPToolKit_MatchingLeftSession:
						hosting = false;
						if (OnLeftSession != null) OnLeftSession(msg);
						return true;
						
					case Messages.MessageType.kNPToolKit_MatchingRoomDestroyed:
						hosting = false;
						if (OnSessionDestroyed != null) OnSessionDestroyed(msg);
						return true;
						
					case Messages.MessageType.kNPToolKit_MatchingKickedOut:
						hosting = false;
						if (OnKickedOut != null) OnKickedOut(msg);
						return true;

					case Messages.MessageType.kNPToolKit_MatchingError:
						if (OnSessionError != null) OnSessionError(msg);
						return true;
				}
				
				return false;
			}
		}
	}
}
